/**
 *
 * \file        error.c
 *
 * \brief       Error handling and rebooting code.
 *
 * \author      Pete McCormick
 *
 * \date        12/6/2007
 *
 * \note
 */

////////////////////////////////////////////////////////////////////////////////

#include "errors.h"
#include "console.h"
#include "os.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "DeviceJoinInterface.h"
#include "StreamMgr.h"
#include "product.h"
#include "cresnet_slave.h"
#include "leds.h"

////////////////////////////////////////////////////////////////////////////////

#define ERROR_LOG_MAX_ENTRIES 255
#define ERR_TIMER_PERIOD_MS 100

////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////

ERRORLOG ErrorLog;
static BOOL s_unsentError;
UINT16 sErrorJoin = 0;

////////////////////////////////////////////////////////////////////////////////

/**
 * \author    Rob Carter
 * \brief     Report any errors that were logged within ISRs
 * \date      3/24/2008
 * \param     unused
 * \return    void
 * \retval    none
**/
void ErrorTimerCallback( UINT32 unused )
{
    if( !s_unsentError )
      return;

    //clear this first
    s_unsentError = false;

    UINT8 CurrentEntry, NumEntries, WrapCount;
    ERRORLOGENTRY *pEntry;

    // get the number of entries currently in the log
    NumEntries = ErrorLog.errCnt;

    // initialize pointer to beginning of Log
    pEntry = ErrorLog.pFirstEntry;

    // if the log is full, it may have wrapped
    if (NumEntries >= ErrorLog.LogSize)
        // set pointer to the oldest entry
        pEntry += ErrorLog.OldestEntry;

    // figure out when to wrap log pointer
    WrapCount = ErrorLog.LogSize - (ErrorLog.OldestEntry +1);

    // for each entry in log
    for(CurrentEntry=0; CurrentEntry < NumEntries; CurrentEntry++)
    {
        //if this entry hasn't been reported
        if( !pEntry->reported )
        {
            //if it doesn't already have a timestamp
            if( pEntry->timestamp == 0 )
                pEntry->timestamp = HwGetMsec();

            // tell switch, join value = subsytem in upper byte, error code in lower one
            sErrorJoin = (pEntry->subsystem << 8) | pEntry->errCode;
            // if cresnet join processing is initialized
            if (pJoinState)
            {
              // report error to controller
              DeviceSendAnalog(SYSTEM_ANALOG_CARD_ERROR, sErrorJoin, STREAM_1);
              pEntry->reported = true;
            }
            DmConsolePrintf("ERR ERROR: subsystem %d, errcode = %d, cause %d\r",
                     pEntry->subsystem, pEntry->errCode, pEntry->causeCode);
        }

        // get pointer to next entry
        if (WrapCount == CurrentEntry)
            pEntry = ErrorLog.pFirstEntry; // reset pointer to beginning of log
        else
            pEntry++; // increment to next entry
    }

    return;
}

void ErrorLogUnlock(void)
{
}


INT32 ErrorLogLock(void)
{
    return 0;
}
UINT32 debugCount;
/**
 * \author    Larry Salant
 * \brief     initialized the error log and allocates memory for it so it can vary by card
 * \date      4/7/2008
 * \param     LogSize : number of entries in the log (each entry is 8 bytes)
 * \return    void
 * \retval    none
**/
void DmErrorInit(UINT8 LogSize)
{
  if (LogSize >= ERROR_LOG_MAX_ENTRIES)
  {
    LogSize = ERROR_LOG_MAX_ENTRIES;
  }
  ErrorLog.pFirstEntry = (ERRORLOGENTRY *)malloc(LogSize * sizeof(ERRORLOGENTRY));
  ErrorLog.LogSize = LogSize;
  ErrorLog.errCnt = 0;
  ErrorLog.OldestEntry = 0;

  s_unsentError = false;
}

/**
 * \author    Rob Carter
 * \brief     Init the error timer
 * \date      3/24/2008
 * \param
 * \return    void
 * \retval    none
**/
void DmErrorTimerInit()
{
  // create a periodic timer
  OsStartTimer( ERR_TIMER_PERIOD_MS, (void*)ErrorTimerCallback, OS_TIMER_PERIODIC );
}

/**
 * \author    Larry Salant
 * \brief     Prints the error log
 * \date      4/7/2008
 * \param     ignore, *cmd : not used (standard params for console commands)
 * \return    void
 * \retval    none
**/
INT32 DmPrintErrorLog(UINT32 ignore, char *cmd)
{
    INT32 result;
    UINT8 CurrentEntry, NumEntries, WrapCount;
    ERRORLOGENTRY *pEntry;

    result = ErrorLogLock();
    if(result != 0)
    {
        DmConsolePrintf("Couldn't lock error log");
    }

    // get the number of entries currently in the log
    NumEntries = ErrorLog.errCnt;
    DmConsolePrintf("%d errors\r", NumEntries);

    // initialize pointer to beginning of Log
    pEntry = ErrorLog.pFirstEntry;

    // if the log is full, it may have wrapped
    if (NumEntries >= ErrorLog.LogSize)
        // set pointer to the oldest entry
        pEntry += ErrorLog.OldestEntry;

    // figure out when to wrap log pointer
    WrapCount = ErrorLog.LogSize - (ErrorLog.OldestEntry +1);

    // for each entry in log
    for(CurrentEntry=0; CurrentEntry < NumEntries; CurrentEntry++)
    {
        // print the entry to the console
        if ( pEntry->causeCode != DM_ERROR_CAUSE_CODE_DO_NOT_PRINT )
            DmConsolePrintError( pEntry->level, pEntry->subsystem, pEntry->errCode, pEntry->causeCode );

        // get pointer to next entry
        if (WrapCount == CurrentEntry)
            pEntry = ErrorLog.pFirstEntry; // reset pointer to beginning of log
        else
            pEntry++; // increment to next entry
    }
    ErrorLogUnlock();
    return 0;
}

/**
 * \author    Larry Salant
 * \brief     clears the error log
 * \date      4/7/2008
 * \param     ignore, *cmd : not used (standard params for console commands)
 * \return    void
 * \retval    none
**/
INT32 DmClearErrorLog(UINT32 ignore, char *cmd)
{
    INT32 result;

    result = ErrorLogLock();
    if(result != 0)
    {
        return -1;
    }
    ErrorLog.OldestEntry = 0;
    ErrorLog.errCnt = 0;
    ErrorLogUnlock();

    // turn off the ERROR LED, if any
    LedControl(LED_ERR, false);

    return 0;
}

void DmReboot(void)
{
#ifdef DEBUG
  while (1)
  {
  }
#else
  DmRebootNotFatal();
#endif

}

/**
* \author         S.Novick
* \brief          Just a wrapper for the fatal case to avoid false positives from Coverity
* \date           2/6/2013
* \param
* \return
* \retval
*/
void DmFatalOnZeroPointer(void* pTest, UINT8 subsystem, UINT8 errCode, UINT16 causeCode)
{
    if (pTest == NULL)
    {
        DmSystemError(DM_ERROR_LEVEL_FATAL, subsystem, errCode, causeCode);
    }
}

/**
 * \author    Larry Salant
 * \brief     Process system errors
 * \date      4/7/2008
 * \param     level - severity of error
 * \param     subsystem - which subsystem reported the error
 * \param     errCode - subsystem dependent error code
 * \param     causeCode - subsystem dependent additional information
 * \return    void
 * \retval    none
**/
void DmSystemError(UINT8 level, UINT8 subsystem, UINT8 errCode, UINT16 causeCode)
{
    UINT8 ErrIndex;
    ERRORLOGENTRY *pEntry;
    UINT8 *pCmdBlock;

    switch (level)
    {
        case DM_ERROR_LEVEL_FATAL:
#ifdef DEBUG
            snprintf(ErrorLog.fatalMsg, sizeof(ErrorLog.fatalMsg),
                     "Fatal Err: sub %d, code %d, cause %d",
                     subsystem, errCode, causeCode);

            DmErrorPrint(ErrorLog.fatalMsg);
#endif
            // turn on the ERROR LED, if any
            LedControl(LED_ERR, true);

			// put error in Command block for bootloader ...
            // let it send the error join on Cresnet
            pCmdBlock = GetBootCmdPtr();

            // turn off interrupts;  we will reboot after this.
            OsEnterCritical();

            // write the upgrade command at the beginning of RAM
            strcpy((char *)pCmdBlock, FATAL_ERROR_MSG);
            // store the cresnet address after the command
            pCmdBlock += FATAL_ERROR_LEN;
            *(UINT8*)pCmdBlock = CresnetGetDefaultId();
            // store the subsystem
            ++pCmdBlock;
            *(UINT8*)pCmdBlock = subsystem;
            // store the error code
            ++pCmdBlock;
            *(UINT8*)pCmdBlock = errCode;

            // fatal error - try to recover
            DmReboot();
            break;
        case DM_ERROR_LEVEL_ERROR:
            if (ErrorLogLock() != 0)
                return;
            // turn on the ERROR LED, if any
            LedControl(LED_ERR, true);

            // The log is a circular buffer, if there is no more room
            ErrIndex = ErrorLog.errCnt;
            if (ErrIndex >= ErrorLog.LogSize)
            {
                // overwrite the oldest entry
                ErrIndex = ErrorLog.OldestEntry;
                // update pointer to oldest
                ErrorLog.OldestEntry++;
                // check if it wrapped
                if (ErrorLog.OldestEntry >= ErrorLog.LogSize)
                    ErrorLog.OldestEntry = 0;
            }
            else // increment the count of errors
                ErrorLog.errCnt++;

            // write error to log
            pEntry = ErrorLog.pFirstEntry + ErrIndex;
            pEntry->level = level;
            pEntry->subsystem = subsystem;
            pEntry->errCode = errCode;
            pEntry->causeCode = causeCode;
            pEntry->reported = false;
            ErrorLogUnlock();

                //this will kill you if you're in an ISR
            if ( !InIsr )
                pEntry->timestamp = HwGetMsec();
            else
                pEntry->timestamp = 0;

                s_unsentError = true;
            break;
        case DM_ERROR_LEVEL_WARNING:
            if ( !InIsr && (causeCode != DM_ERROR_CAUSE_CODE_DO_NOT_PRINT) )
                DmConsolePrintError( level, subsystem, errCode, causeCode );

            break;
        case DM_ERROR_LEVEL_NOTICE:
            if ( !InIsr && (causeCode != DM_ERROR_CAUSE_CODE_DO_NOT_PRINT) )
                DmConsolePrintError( level, subsystem, errCode, causeCode );
            break;
        case DM_ERROR_LEVEL_DEBUG:
#ifdef DEBUG
            if ( !InIsr && (causeCode != DM_ERROR_CAUSE_CODE_DO_NOT_PRINT) )
                DmConsolePrintError( level, subsystem, errCode, causeCode );
#endif
            break;
        default:
            break;
    }
}
/**
 * \author    Larry Salant
 * \brief     Process All clear response - report last error
 * \date      4/7/2008
 * \param     none
 * \return    void
 * \retval    none
**/
void DmErrorProcessAllClear( UINT8 numStreams )
{
    for( int i=0; i<numStreams; i++ )
    {
        DeviceSendAnalog(SYSTEM_ANALOG_CARD_ERROR, 0, DEVICE_JOIN_INDEX( i, 0, TOP_LEVEL_SLOT_0) );
    }
}

INT32 NvramRebootCmd(UINT32 ignore, char * cmd)
{
    return 0;
}

void DmSendCyclePower(void)
{
    DeviceSendDigital( SYSTEM_DIGITAL_REBOOT, true, 0 );
}

